const AttendanceApp = (function(){
  const api = {
    getEmployees: 'api/get_employees.php',
    saveEmployee: 'api/save_employee.php',
    deleteEmployee: 'api/delete_employee.php',
    getAttendance: 'api/get_attendance.php',
    saveAttendance: 'api/save_attendance.php',
    exportCsv: 'api/export_csv.php'
  };

  // helper
  function qs(sel) { return document.querySelector(sel); }
  function qsa(sel) { return Array.from(document.querySelectorAll(sel)); }

  async function getJSON(url, opts){
    const res = await fetch(url, opts);
    return res.json();
  }

  // STAFF
  async function initStaff(){
    await loadEmployeesToSelect();
    qs('#btnIn').addEventListener('click', ()=> mark('in'));
    qs('#btnOut').addEventListener('click', ()=> mark('out'));
    qs('#empSelect').addEventListener('change', loadHistoryForSelected);
    // load history for first employee
    loadHistoryForSelected();
  }

  async function loadEmployeesToSelect(){
    const list = await getJSON(api.getEmployees);
    const sel = qs('#empSelect'); sel.innerHTML='';
    if(list.length===0){ sel.innerHTML='<option value="">No employees</option>'; return; }
    list.forEach(e=>{
      const opt = document.createElement('option'); opt.value = e.id; opt.textContent = e.name + ' — ' + e.emp_id; sel.appendChild(opt);
    });
  }

  async function mark(type){
    const emp = qs('#empSelect').value;
    const msg = qs('#msg'); msg.textContent='Saving...';
    const fd = new FormData(); fd.append('emp_id', emp); fd.append('type', type);
    const res = await fetch(api.saveAttendance, {method:'POST', body:fd});
    const j = await res.json();
    if (j.ok) msg.textContent = 'Saved: ' + j.record.name + ' - ' + j.record.type + ' at ' + j.record.time + ' (IP: ' + (j.record.ip||'') + ')';
    else msg.textContent = j.msg || 'Error';
    loadHistoryForSelected();
  }

  async function loadHistoryForSelected(){
    const emp = qs('#empSelect')?.value;
    const tbl = qs('#historyTable tbody'); tbl.innerHTML='';
    const att = await getJSON(api.getAttendance);
    const recent = att.filter(a=>!emp || a.emp_id===emp).slice(-20).reverse();
    recent.forEach(r=>{
      const tr = document.createElement('tr');
      tr.innerHTML = `<td>${r.date}</td><td>${r.time}</td><td>${r.type}</td>`;
      tbl.appendChild(tr);
    });
  }

  // ADMIN
  async function initAdmin(){
    await renderEmployeesTable();
    qs('#addEmp').addEventListener('click', addEmployeeHandler);
    qs('#filterBtn').addEventListener('click', renderAttendanceTable);
    qs('#exportBtn').addEventListener('click', exportCsv);
    renderAttendanceTable();
  }

  async function renderEmployeesTable(){
    const list = await getJSON(api.getEmployees);
    const tbody = qs('#empTable tbody'); tbody.innerHTML='';
    list.forEach(e=>{
      const tr = document.createElement('tr');
      tr.innerHTML = `<td>${e.name}</td><td>${e.emp_id}</td><td>${e.role||''}</td>
        <td>
          <button class="btn small edit" data-id="${e.id}">Edit</button>
          <button class="btn small del" data-id="${e.id}">Delete</button>
        </td>`;
      tbody.appendChild(tr);
    });
    qsa('.edit').forEach(b=>b.addEventListener('click', onEdit));
    qsa('.del').forEach(b=>b.addEventListener('click', onDelete));
  }

  async function addEmployeeHandler(){
    const name = qs('#empName').value.trim();
    const code = qs('#empCode').value.trim();
    const role = qs('#empRole').value.trim();
    if(!name||!code){ qs('#empMsg').textContent='Name and code required'; return; }
    const fd = new FormData(); fd.append('action','add'); fd.append('name',name); fd.append('emp_id',code); fd.append('role',role);
    const res = await fetch(api.saveEmployee,{method:'POST',body:fd});
    const j = await res.json();
    if(j.ok){ qs('#empMsg').textContent = 'Added'; qs('#empName').value=''; qs('#empCode').value=''; qs('#empRole').value=''; renderEmployeesTable(); loadEmployeesToSelect(); }
    else qs('#empMsg').textContent = j.msg||'Error';
  }

  async function onEdit(e){
    const id = e.currentTarget.dataset.id;
    // simple prompt edit
    const list = await getJSON(api.getEmployees);
    const emp = list.find(x=>x.id===id);
    if(!emp) return alert('not found');
    const newName = prompt('Name', emp.name);
    if(newName===null) return;
    const newCode = prompt('Employee code', emp.emp_id);
    const newRole = prompt('Role', emp.role||'');
    const fd = new FormData(); fd.append('action','edit'); fd.append('id', id); fd.append('name', newName); fd.append('emp_id', newCode); fd.append('role', newRole);
    const res = await fetch(api.saveEmployee,{method:'POST',body:fd}); const j = await res.json();
    if(j.ok){ renderEmployeesTable(); loadEmployeesToSelect(); }
    else alert(j.msg||'Error');
  }

  async function onDelete(e){
    const id = e.currentTarget.dataset.id;
    if(!confirm('Delete employee?')) return;
    const fd = new FormData(); fd.append('id', id);
    const res = await fetch(api.deleteEmployee,{method:'POST',body:fd}); const j = await res.json();
    if(j.ok){ renderEmployeesTable(); loadEmployeesToSelect(); } else alert(j.msg||'Error');
  }

  async function renderAttendanceTable(){
    const start = qs('#startDate').value; const end = qs('#endDate').value;
    const att = await getJSON(api.getAttendance);
    const rows = att.filter(r => (start? r.date>=start : true) && (end? r.date<=end : true));
    const tbody = qs('#attTable tbody'); tbody.innerHTML = '';
    rows.reverse().forEach(r=>{
      const tr = document.createElement('tr');
      tr.innerHTML = `<td>${r.date}</td><td>${r.time}</td><td>${r.emp_code}</td><td>${r.name}</td><td>${r.type}</td><td>${r.ip||''}</td>`;
      tbody.appendChild(tr);
    });
  }

  function exportCsv(){
    const start = qs('#startDate').value; const end = qs('#endDate').value;
    let url = api.exportCsv;
    const params = [];
    if(start) params.push('start='+encodeURIComponent(start));
    if(end) params.push('end='+encodeURIComponent(end));
    if(params.length) url += '?' + params.join('&');
    window.location = url;
  }

  return { initStaff, initAdmin };
})();